/*!
 * Copyright (c) 2015, Curtis Bratton
 * All rights reserved.
 * LIDERNetwork Tecnologia
 * Adaptador por Carlos Jorge M. JR
 * Liquid Fill Gauge v2.0 - MODERNIZADO
 * Alterações realizadas por Carlos Jorge Junior by LIDERNetwork
 */

// ===== VERIFICAÇÃO DE DEPENDÊNCIAS =====
(function() {
    // Verificar se D3.js está carregado
    if (typeof d3 === 'undefined') {
        console.warn('⚠️ D3.js não encontrado. Carregue D3.js antes do tanks.js');
        return;
    }
    
    // Verificar se jQuery está carregado (opcional para este script)
    if (typeof jQuery === 'undefined') {
        console.info('ℹ️ jQuery não encontrado, mas não é necessário para os tanques');
    }
    
    console.log('✅ Dependências verificadas - Tanks.js carregado');
})();

// ===== CORES MODERNAS E GRADIENTES =====
function getModernWaveColor(waveText) {
    var keywords = waveText.split(' ');
    for (var i = 0; i < keywords.length; i++) {
        switch (keywords[i].toUpperCase()) {
            case 'COMUM':
                return {
                    primary: '#f59e0b',    // Amber moderno
                    secondary: '#d97706',  // Amber escuro
                    gradient: 'url(#gradientComum)'
                };
            case 'ETANOL':
                return {
                    primary: '#06b6d4',    // Cyan moderno
                    secondary: '#0891b2',  // Cyan escuro
                    gradient: 'url(#gradientEtanol)'
                };
            case 'S10':
                return {
                    primary: '#a16207',    // Marrom claro/amber escuro
                    secondary: '#92400e',  // Marrom mais escuro
                    gradient: 'url(#gradientS10)'
                };
            case 'S500':
                return {
                    primary: '#6b7280',    // Gray moderno
                    secondary: '#4b5563',  // Gray escuro
                    gradient: 'url(#gradientS500)'
                };
            case 'ADITIVADA':
                return {
                    primary: '#10b981',    // Emerald moderno
                    secondary: '#059669',  // Emerald escuro
                    gradient: 'url(#gradientAditivada)'
                };
        }
    }
    return {
        primary: '#f59e0b',
        secondary: '#d97706',
        gradient: 'url(#gradientComum)'
    };
}

// ===== CONFIGURAÇÕES MODERNAS =====
function liquidFillGaugeDefaultSettings(){
    return {
        minValue: 0,
        maxValue: 100,
        circleThickness: 0.04,        // Borda mais espessa para ser visível
        circleFillGap: 0.03,          // Gap menor
        circleColor: "#e5e7eb",       // Cinza bem claro para suavidade
        waveHeight: 0.1,              // Ondas mais pronunciadas
        waveCount: 1.5,               // Menos ondas, mais fluidas
        waveRiseTime: 3000,           // Animação mais suave
        waveAnimateTime: 8000,        // Animação mais lenta e hipnotizante
        waveRise: true,
        waveHeightScaling: true,
        waveAnimate: true,
        waveColor: "#f59e0b",
        waveOffset: 0.8,              // Offset mais suave
        textVertPosition: 0.5,
        textSize: 0.9,                // Texto menor (era 1.1)
        valueCountUp: true,
        displayPercent: true,
        textColor: "#374151",         // Texto mais escuro
        waveTextColor: "#ffffff",     // Texto branco sobre líquido
        shadowEnabled: true,          // Nova funcionalidade
        glowEffect: true,             // Nova funcionalidade
        modernGradients: true         // Nova funcionalidade
    };
}

// ===== FUNÇÃO PRINCIPAL MODERNIZADA =====
function loadLiquidFillGauge(elementId, value, produto) {
    // Verificação de segurança
    if (typeof d3 === 'undefined') {
        console.error('❌ D3.js é necessário para os tanques funcionarem');
        return;
    }
    
    if (!elementId || !document.getElementById(elementId)) {
        console.error('❌ Elemento não encontrado:', elementId);
        return;
    }
    
    var config = liquidFillGaugeDefaultSettings();
    var colorScheme = getModernWaveColor(produto || 'COMUM');
    
    config.waveColor = colorScheme.primary;
    
    console.log("🔧 Produto:", produto);
    console.log("🎨 Esquema de cores:", colorScheme);
    
    var gauge = d3.select("#" + elementId);
    var radius = Math.min(parseInt(gauge.style("width")), parseInt(gauge.style("height")))/2;
    var locationX = parseInt(gauge.style("width"))/2 - radius;
    var locationY = parseInt(gauge.style("height"))/2 - radius;
    var fillPercent = Math.max(config.minValue, Math.min(config.maxValue, value))/config.maxValue;

    // ===== ESCALAS APRIMORADAS =====
    var waveHeightScale;
    if(config.waveHeightScaling){
        waveHeightScale = d3.scale.linear()
            .range([0, config.waveHeight, 0])
            .domain([0, 50, 100]);
    } else {
        waveHeightScale = d3.scale.linear()
            .range([config.waveHeight, config.waveHeight])
            .domain([0, 100]);
    }

    var textPixels = (config.textSize * radius / 2);
    var textFinalValue = parseFloat(value).toFixed(1); // Uma casa decimal
    var textStartValue = config.valueCountUp ? config.minValue : textFinalValue;
    var percentText = config.displayPercent ? "%" : "";
    var circleThickness = config.circleThickness * radius;
    var circleFillGap = config.circleFillGap * radius;
    var fillCircleMargin = circleThickness + circleFillGap;
    var fillCircleRadius = radius - fillCircleMargin;
    var waveHeight = fillCircleRadius * waveHeightScale(fillPercent * 100);

    var waveLength = fillCircleRadius * 2 / config.waveCount;
    var waveClipCount = 1 + config.waveCount;
    var waveClipWidth = waveLength * waveClipCount;

    // ===== FUNÇÃO DE ARREDONDAMENTO MELHORADA =====
    var textRounder = function(value){ 
        return parseFloat(value).toFixed(0); // Sempre uma casa decimal
    };

    // ===== DADOS DA ONDA COM SUAVIZAÇÃO =====
    var data = [];
    for(var i = 0; i <= 60 * waveClipCount; i++){ // Mais pontos para suavidade
        data.push({
            x: i/(60 * waveClipCount), 
            y: (i/(60))
        });
    }

    // ===== ESCALAS DE DESENHO =====
    var gaugeCircleX = d3.scale.linear().range([0, 2*Math.PI]).domain([0, 1]);
    var gaugeCircleY = d3.scale.linear().range([0, radius]).domain([0, radius]);
    var waveScaleX = d3.scale.linear().range([0, waveClipWidth]).domain([0, 1]);
    var waveScaleY = d3.scale.linear().range([0, waveHeight]).domain([0, 1]);

    var waveRiseScale = d3.scale.linear()
        .range([(fillCircleMargin + fillCircleRadius * 2 + waveHeight), 
                (fillCircleMargin - waveHeight)])
        .domain([0, 1]);
        
    var waveAnimateScale = d3.scale.linear()
        .range([0, waveClipWidth - fillCircleRadius * 2])
        .domain([0, 1]);

    var textRiseScaleY = d3.scale.linear()
        .range([fillCircleMargin + fillCircleRadius * 2, 
                (fillCircleMargin + textPixels * 0.7)])
        .domain([0, 1]);

    // ===== GRUPO PRINCIPAL =====
    var gaugeGroup = gauge.append("g")
        .attr('transform', 'translate(' + locationX + ',' + locationY + ')');

    // ===== DEFINIÇÕES DE GRADIENTES MODERNOS =====
    if(config.modernGradients) {
        var defs = gaugeGroup.append("defs");
        
        // Gradiente do líquido
        var liquidGradient = defs.append("linearGradient")
            .attr("id", "liquidGradient" + elementId)
            .attr("gradientUnits", "userSpaceOnUse")
            .attr("x1", "0%").attr("y1", "0%")
            .attr("x2", "0%").attr("y2", "100%");
            
        liquidGradient.append("stop")
            .attr("offset", "0%")
            .attr("stop-color", colorScheme.primary)
            .attr("stop-opacity", 0.8);
            
        liquidGradient.append("stop")
            .attr("offset", "100%")
            .attr("stop-color", colorScheme.secondary)
            .attr("stop-opacity", 1);

        // Gradiente do círculo externo (borda muito suave)
        var circleGradient = defs.append("linearGradient")
            .attr("id", "circleGradient" + elementId)
            .attr("gradientUnits", "userSpaceOnUse")
            .attr("x1", "0%").attr("y1", "0%")
            .attr("x2", "100%").attr("y2", "100%");
            
        circleGradient.append("stop")
            .attr("offset", "0%")
            .attr("stop-color", "#f9fafb");  // Cinza muito claro
            
        circleGradient.append("stop")
            .attr("offset", "100%")
            .attr("stop-color", "#e5e7eb");  // Cinza claro

        // Filtro de sombra
        if(config.shadowEnabled) {
            var filter = defs.append("filter")
                .attr("id", "shadow" + elementId)
                .attr("x", "-50%")
                .attr("y", "-50%")
                .attr("width", "200%")
                .attr("height", "200%");

            filter.append("feDropShadow")
                .attr("dx", "0")
                .attr("dy", "4")
                .attr("stdDeviation", "8")
                .attr("flood-color", "rgba(0,0,0,0.15)");
        }

        // Filtro de brilho
        if(config.glowEffect) {
            var glowFilter = defs.append("filter")
                .attr("id", "glow" + elementId)
                .attr("x", "-50%")
                .attr("y", "-50%")
                .attr("width", "200%")
                .attr("height", "200%");

            var feGaussianBlur = glowFilter.append("feGaussianBlur")
                .attr("stdDeviation", "3")
                .attr("result", "coloredBlur");

            var feMerge = glowFilter.append("feMerge");
            feMerge.append("feMergeNode").attr("in", "coloredBlur");
            feMerge.append("feMergeNode").attr("in", "SourceGraphic");
        }
    }

    // ===== CÍRCULO EXTERNO MODERNIZADO =====
    var gaugeCircleArc = d3.svg.arc()
        .startAngle(gaugeCircleX(0))
        .endAngle(gaugeCircleX(1))
        .outerRadius(gaugeCircleY(radius))
        .innerRadius(gaugeCircleY(radius - circleThickness));
        
    var outerCircle = gaugeGroup.append("path")
        .attr("d", gaugeCircleArc)
        .style("fill", config.modernGradients ? "url(#circleGradient" + elementId + ")" : config.circleColor)
        .style("stroke", "#d1d5db")        // Borda cinza bem clara
        .style("stroke-width", "1px")      // Linha fina na borda
        .attr('transform', 'translate(' + radius + ',' + radius + ')');

    if(config.shadowEnabled) {
        outerCircle.style("filter", "url(#shadow" + elementId + ")");
    }

    // ===== TEXTO PRINCIPAL MODERNIZADO =====
    var text1 = gaugeGroup.append("text")
        .text(textRounder(textStartValue) + percentText)
        .attr("class", "liquidFillGaugeText")
        .attr("text-anchor", "middle")
        .attr("font-size", textPixels + "px")
        .attr("font-family", "'Inter', -apple-system, BlinkMacSystemFont, sans-serif")
        .attr("font-weight", "600")
        .style("fill", config.textColor)
        .attr('transform', 'translate(' + radius + ',' + textRiseScaleY(config.textVertPosition) + ')');

    // ===== ÁREA DE CLIPPING DA ONDA MELHORADA =====
    var clipArea = d3.svg.area()
        .x(function(d) { return waveScaleX(d.x); })
        .y0(function(d) { 
            return waveScaleY(
                Math.sin(Math.PI * 2 * config.waveOffset * -1 + 
                        Math.PI * 2 * (1 - config.waveCount) + 
                        d.y * 2 * Math.PI)
            ); 
        })
        .y1(function(d) { return (fillCircleRadius * 2 + waveHeight); })
        .interpolate("cardinal"); // Interpolação suave

    var waveGroup = gaugeGroup.append("defs")
        .append("clipPath")
        .attr("id", "clipWave" + elementId);
        
    var wave = waveGroup.append("path")
        .datum(data)
        .attr("d", clipArea)
        .attr("T", 0);

    // ===== CÍRCULO INTERNO COM LÍQUIDO =====
    var fillCircleGroup = gaugeGroup.append("g")
        .attr("clip-path", "url(#clipWave" + elementId + ")");
        
    var innerCircle = fillCircleGroup.append("circle")
        .attr("cx", radius)
        .attr("cy", radius)
        .attr("r", fillCircleRadius)
        .style("fill", config.modernGradients ? "url(#liquidGradient" + elementId + ")" : config.waveColor);

    if(config.glowEffect) {
        innerCircle.style("filter", "url(#glow" + elementId + ")");
    }

    // ===== TEXTO SOBRE O LÍQUIDO =====
    var text2 = fillCircleGroup.append("text")
        .text(textRounder(textStartValue) + percentText)
        .attr("class", "liquidFillGaugeText")
        .attr("text-anchor", "middle")
        .attr("font-size", textPixels + "px")
        .attr("font-family", "'Inter', -apple-system, BlinkMacSystemFont, sans-serif")
        .attr("font-weight", "700")
        .style("fill", config.waveTextColor)
        .style("text-shadow", "0 1px 2px rgba(0,0,0,0.3)")
        .attr('transform', 'translate(' + radius + ',' + textRiseScaleY(config.textVertPosition) + ')');

    // ===== ANIMAÇÃO DE CONTAGEM MELHORADA =====
    if(config.valueCountUp){
        var textTween = function(){
            var i = d3.interpolate(this.textContent, textFinalValue);
            return function(t) { 
                this.textContent = textRounder(i(t)) + percentText; 
            }
        };
        
        text1.transition()
            .duration(config.waveRiseTime)
            .ease("ease-out")
            .tween("text", textTween);
            
        text2.transition()
            .duration(config.waveRiseTime)
            .ease("ease-out")
            .tween("text", textTween);
    }

    // ===== ANIMAÇÃO DA ONDA APRIMORADA =====
    var waveGroupXPosition = fillCircleMargin + fillCircleRadius * 2 - waveClipWidth;
    
    if(config.waveRise){
        waveGroup.attr('transform', 'translate(' + waveGroupXPosition + ',' + waveRiseScale(0) + ')')
            .transition()
            .duration(config.waveRiseTime)
            .ease("ease-out")
            .attr('transform', 'translate(' + waveGroupXPosition + ',' + waveRiseScale(fillPercent) + ')')
            .each("start", function(){ 
                wave.attr('transform', 'translate(1,0)'); 
            });
    } else {
        waveGroup.attr('transform', 'translate(' + waveGroupXPosition + ',' + waveRiseScale(fillPercent) + ')');
    }

    if(config.waveAnimate) animateWave();

    // ===== FUNÇÃO DE ANIMAÇÃO CONTÍNUA =====
    function animateWave() {
        wave.attr('transform', 'translate(' + waveAnimateScale(wave.attr('T')) + ',0)');
        wave.transition()
            .duration(config.waveAnimateTime * (1 - wave.attr('T')))
            .ease('linear')
            .attr('transform', 'translate(' + waveAnimateScale(1) + ',0)')
            .attr('T', 1)
            .each('end', function(){
                wave.attr('T', 0);
                animateWave(config.waveAnimateTime);
            });
    }

    // ===== FUNÇÃO DE ATUALIZAÇÃO MELHORADA =====
    function GaugeUpdater(){
        this.update = function(value){
            var newFinalValue = parseFloat(value).toFixed(1);
            var textRounderUpdater = function(value){ 
                return parseFloat(value).toFixed(1); 
            };

            var textTween = function(){
                var i = d3.interpolate(this.textContent, parseFloat(value).toFixed(1));
                return function(t) { 
                    this.textContent = textRounderUpdater(i(t)) + percentText; 
                }
            };

            text1.transition()
                .duration(config.waveRiseTime)
                .ease("ease-out")
                .tween("text", textTween);
                
            text2.transition()
                .duration(config.waveRiseTime)
                .ease("ease-out")
                .tween("text", textTween);

            var fillPercent = Math.max(config.minValue, Math.min(config.maxValue, value))/config.maxValue;
            var waveHeight = fillCircleRadius * waveHeightScale(fillPercent * 100);
            var waveRiseScale = d3.scale.linear()
                .range([(fillCircleMargin + fillCircleRadius * 2 + waveHeight), 
                        (fillCircleMargin - waveHeight)])
                .domain([0, 1]);
                
            var newHeight = waveRiseScale(fillPercent);
            var waveScaleX = d3.scale.linear().range([0, waveClipWidth]).domain([0, 1]);
            var waveScaleY = d3.scale.linear().range([0, waveHeight]).domain([0, 1]);
            
            var newClipArea;
            if(config.waveHeightScaling){
                newClipArea = d3.svg.area()
                    .x(function(d) { return waveScaleX(d.x); })
                    .y0(function(d) { 
                        return waveScaleY(
                            Math.sin(Math.PI * 2 * config.waveOffset * -1 + 
                                    Math.PI * 2 * (1 - config.waveCount) + 
                                    d.y * 2 * Math.PI)
                        ); 
                    })
                    .y1(function(d) { return (fillCircleRadius * 2 + waveHeight); })
                    .interpolate("cardinal");
            } else {
                newClipArea = clipArea;
            }

            var newWavePosition = config.waveAnimate ? waveAnimateScale(1) : 0;
            
            wave.transition()
                .duration(0)
                .transition()
                .duration(config.waveAnimate ? 
                         (config.waveAnimateTime * (1 - wave.attr('T'))) : 
                         (config.waveRiseTime))
                .ease('ease-out')
                .attr('d', newClipArea)
                .attr('transform', 'translate(' + newWavePosition + ',0)')
                .attr('T', '1')
                .each("end", function(){
                    if(config.waveAnimate){
                        wave.attr('transform', 'translate(' + waveAnimateScale(0) + ',0)');
                        animateWave(config.waveAnimateTime);
                    }
                });
                
            waveGroup.transition()
                .duration(config.waveRiseTime)
                .ease("ease-out")
                .attr('transform', 'translate(' + waveGroupXPosition + ',' + newHeight + ')');
        }
    }

    console.log("✅ Tanque modernizado carregado:", elementId);
    return new GaugeUpdater();
}